using System;
using System.Diagnostics;

namespace Implab {
    class PromiseFuncReaction<TIn, TRet> : IResolvable<TIn> {
        readonly Deferred<TRet> m_next;

        readonly IDispatcher m_dispatcher;

        readonly Action<TIn, Deferred<TRet>> m_fulfilled;

        readonly Action<Exception, Deferred<TRet>> m_rejected;

        public IPromise<TRet> Promise {
            get { return m_next.Promise; }
        }

        public PromiseFuncReaction(Action<TIn, Deferred<TRet>> fulfilled, Action<Exception, Deferred<TRet>> rejected, Deferred<TRet> next, IDispatcher dispatcher) {
            m_next = next;
            m_fulfilled = fulfilled;
            m_rejected = rejected;
            m_dispatcher = dispatcher;
        }

        public void Resolve(TIn result) {
            if (m_fulfilled != null) {
                if (m_dispatcher != null)
                    m_dispatcher.Enqueue(ResolveImpl, result);
                else
                    ResolveImpl(result);
            } else {
                try {
                    m_next.Resolve((TRet)(object)result);
                } catch(Exception error) {
                    // handle cast exceptions
                    m_next.Reject(error);
                }
            }
        }

        void ResolveImpl (TIn result) {
            m_fulfilled(result, m_next);
        }

        public void Reject(Exception error) {
            if (m_fulfilled != null) {
                if (m_dispatcher != null)
                    m_dispatcher.Enqueue(RejectImpl, error);
                else
                    RejectImpl(error);
            } else {
                m_next.Reject(error);
            }
        }

        void RejectImpl(Exception error) {
            m_rejected(error, m_next);
        }


        public static PromiseFuncReaction<TIn,TRet> Create(Func<TIn,TRet> fulfilled, Func<Exception, TRet> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TIn,TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }

        public static PromiseFuncReaction<TIn,TRet> Create(Func<TIn,IPromise<TRet>> fulfilled, Func<Exception, TRet> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TIn,TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }

        public static PromiseFuncReaction<TIn,TRet> Create(Func<TIn,TRet> fulfilled, Func<Exception, IPromise<TRet>> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TIn,TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }

        public static PromiseFuncReaction<TIn,TRet> Create(Func<TIn,IPromise<TRet>> fulfilled, Func<Exception, IPromise<TRet>> rejected, IDispatcher dispatcher) {
            return new PromiseFuncReaction<TIn,TRet>(
                fulfilled != null ? PromiseHandler.Create(fulfilled) : null,
                rejected != null ? PromiseHandler.Create(rejected) : null,
                new Deferred<TRet>(),
                dispatcher
            );
        }
    }
}